#!/usr/bin/env ruby
# frozen_string_literal: false
#
#  Tk::Img demo
#
#    --  This script is based on demo.tcl of Tcl/Tk's 'Img' extension.
#        Image data in this script is those of demo.tcl.
#        Please read 'license_terms_of_Img_extension' file.
#
require 'tk'
require 'tkextlib/tkimg'

#
# Make the Image format available.
#
class TkImg_demo
  def initialize
    img_version = Tk::Img.package_version

    @typeFrame = Hash.new
    @imgPriv = Hash.new

    root = TkRoot.new(:title=>'Tests for available image formats')

    root.winfo_children.each{|w| w.destroy}
    TkImage.names{|img| img.delete}

    f = TkFrame.new
    TkButton.new(f, :text=>'Dismiss', :command=>proc{exit}).pack(:side=>:left)
    f.pack(:side=>:top, :expand=>:y, :fill=>:both)

    TkMessage.new(:aspect=>900, :text=>format('This page shows the available image formats of the Img extension (Img version %s, using Tcl/Tk %s)', img_version, Tk::TK_PATCHLEVEL)).pack(:side=>:top, :expand=>:y, :fill=>:both)
  end

##############################

  def update_animated_gif(w, method, num)
    return unless @imgPriv[w]

    if @imgPriv[w][:args]
      im = TkPhotoImage.new
      im.copy(@imgPriv[w][num])
      num += 1
      begin
        im.configure(@imgPriv[w][:args].merge(:format=>[:gif, {:index=>num}]))
        im.configure(:data=>'', :file=>'') #free storage
        @imgPriv[w][num] = im
      rescue
        @imgPriv[w].delete(:args)
        if num > 1
          num = 0
          im = @imgPriv[w][num]
        else
          # this is not an animated GIF; just stop
          @imgPriv[w].delete(0)
          return
        end
      end
    else
      num += 1
      num = 0 unless @imgPriv[w][num]
      im = @imgPriv[w][num]
    end
    begin
      w.__send__(method, im)
      Tk.update_idletasks
      Tk.after(20, proc{update_animated_gif(w, method, num)})
    rescue
      @imgPriv[w].delete(:args)
      @imgPriv[w].each{|im|
        @im.delete
        @imgPriv.delete(im)
      }
    end
  end

  def show_animated_gif(keys)
    w = TkLabel.new
    begin
      im = TkPhotoImage.new(keys.merge(:format=>[:gif, {:index=>0}]))
      im.configure(:data=>'', :file=>'', :format=>'') #free storage
      w.image(im)
      @imgPriv[w] ||= Hash.new
      @imgPriv[w][0] = im
      @imgPriv[w][:args] = keys
      Tk.update_idletasks
      Tk.after(20, proc{update_animated_gif(w, :image, 0)})
    rescue => e
      w.configure(:text=>"error displaying animated gif:\n#{e.message}",
                  :image=>'', :relief=>:ridge)
    end
    w.pack
  end

  def show_image(fmt, type, data)
    fmt = fmt.to_s.capitalize
    unless @typeFrame[fmt]
      @typeFrame[fmt] = TkFrame.new.pack(:side=>:top, :expand=>true, :fill=>:x)
      TkLabel.new(@typeFrame[fmt], :text=>"#{fmt} :  ").pack(:side=>:left)
    end
    begin
      f = TkFrame.new(@typeFrame[fmt],
                      :borderwidth=>2, :relief=>:ridge).pack(:side=>:left)
      im = TkPhotoImage.new(:data=>data)
      im['data'] = ''
      TkLabel.new(f, :image=>im).pack
      TkLabel.new(f, :text=>type, :borderwidth=>0, :pady=>0, :padx=>2,
                  :font=>'Helvetica 8').pack
    rescue => e
      TkMessage.new(f, :text=>"error displaying #{type} image: #{e.message}",
                    :aspect=>250).pack
    end
    Tk.update
  end

end

##############

demo = TkImg_demo.new

##############

demo.show_animated_gif(:data=><<'ENDofIMG')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=
ENDofIMG

demo.show_image('bmp', '1-bit', <<'ENDofIMG')
Qk3OAQAAAAAAAD4AAAAoAAAAIgAAADIAAAABAAEAAAAAAJABAABtCwAAbQsAAAIAAAACAAAA
MzPMAMwzMwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAZmTKAAAAAABIRIwAAAAAAEZAigAAAAAAQECIAAAAAAAAAAgAAAAAAAAEAA
AAAAAAABQAAAAAAAAAPAAAAAAAAAB+AAAAAAAAAC4AAAAAAAAALgAAAAAAAAAsAAAAAAAAAC
wAAAAAAAAALgAAAAAAAABuAAAAAAAAAG4AAAAAAAAAbgAAAAAAAABuAAAAAAAAAG4AAAAAAA
AAfgAAAAAAAAA+AAAAAAAAADcAAAAAAAAAPwAAAAAAAAA/AAAAAAAAAD+AAAAAAAAAP4AAAA
AAAAA/gAAAAAAAAD4AAAAAAAAAHgAAAAAAAAAfAAAAAAAAAA+AAAAAAAAAB8AAAAAAAAADgA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
ENDofIMG

demo.show_image('bmp', '4-bit', <<'ENDofIMG')
Qk1aBAAAAAAAAHIAAAAoAAAAIgAAADIAAAABAAQAAAAAAOgDAABtCwAAbQsAAA8AAAAPAAAA
////AAD//wDMzP8AmZn/AGaZ/wAzM8wAmZnMAGaZzAAzZswAzMyZAJmZmQDMZjMAZjNmADMz
ZgBmMzMABWMgAAAAAAAAAAAAAAAAAAAAAAAFVVVVVVVVVVVVVVVVVVVVUAAAAAVVVVVVVVVV
VVVVVVVVVVVQAAAABVVVVVVVVVVVVVVVVVVVVVAAAAAFVVVVVVVVVVVVVVVVVVVVUAAAAAVV
VVVVVVVVVVVVVVVVVVWAAAAABVVVVVVVVVVVVVVVVVVVVVAAAAAFVVBVU3VSVVNVB1UFBVVV
UAAAAAVVUFU1VVNVVVUFVQBVVVVgAAAABVVQVVUFU1VVVQVVBVVVVTAAAAAIVVBVVVVTVXVV
BVUFVVVVMAAAAAVVVVVVVVFVFVVVVQVVVVUgAAAABVVVVVVVVVUFVVVVVVVVVSAAAAAHVVVV
VVVVW5tVVVVVVVVVAAAAAAZVVVVVVVW8u2VVVVVVVVUAAAAAA1VVVVVVW7W7tVVVVVVVVQAA
AAADVVVVVVWCtbuzVVVVVVVVAAAAAANVVVVVVVW1vrJVVVVVVVUAAAAAAlVVVVVVWLW7UFVV
VVVVVQAAAAACVVVVVVVatbtVVVVVVVVVAAAAAABVVVVVVVq3u8NVVVVVVVUAAAAAAFVVVVVV
XrS74VVVVVVVVQAAAAAAVVVVVVWLtbuyVVVVVVVVAAAAAABVVVVVVUu1u7BVVVVVVVUAAAAA
AFVVVVVVO7W7sFVVVVVVVQAAAAAAVVVVVVUbtbu6VVVVVVVVAAAAAABVVVVVVSu9u1VVVVVV
VVUAAAAAAFVVVVVVJb67tVVVVVVVVQAAAAAAVVVVVVVVu6u7VVVVVVVVAAAAAABVVVVVVVW7
u7slVVVVVVUAAAAAAFVVVVVVVbu7u1VVVVVVVQAAAAACVVVVVVVVu7u7wlRVVVVVAAAAAAJV
VVVVVVW7u7uwV1VVVVUAAAAAA1VVVVVVVbu7u7BVhVVVVSAAAAADVVVVVYVVu7vFAFUVVVVV
MAAAAAZVVVVVhVUbu7VVVRVXVVVgAAAAB1VVVVVVVVu7u1VVFVFVVVAAAAAFVVVVVVhVVLu7
xVVVVVVVgAAAAAVVVVVVUVVVK7u1VVVVVVVQAAAACFVVVVhRVVVQq7VVVVVVVVAAAAAFVVVV
VFV1VVUgBVVVVVVVUAAAAAVVVVVUVRVVVVVVVVVVVVVQAAAABVVVVViFUVVVVVVVVVVVVVAA
AAAFVVVVVRVUFVVVVVVVVVVVUAAAAAVVVVVVVVVVVVVVVVVVVVVQAAAABVVVVVVVVVVVVVVV
VVVVVVIAAAAFVVVVVVVVVVVVVVVVVVVVUQAAAAVVVVVVVVVVVVVVVTdVVVVWAAAAAAAhNnWF
VVVVVVcAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA==}
ENDofIMG

if false
demo.show_image('bmp', '4-bit-RLE', <<'ENDofIMG')
Qk1sAwAAAAAAAHYAAAAoAAAAIgAAADIAAAABAAQAAgAAAPYCAAAAAAAAAAAAABAAAAAQAAAA
////AMzM/wAA//8AmZn/AJmZzABmmcwAMzPMADNmzABmmf8AmZmZAMxmMwBmM2YAZjMzADMz
ZgDMzJkAAAAAAAAGBkMQABwAAAACBh5mAmAAAAIGHmYCYAAAAgYeZgJgAAACBh5mAmAAAAIG
HmYCcAAAAgYeZgJgAAAAIgZmYGZjVmFmY2YFZgYGZmZgAAAAACIGZmBmNmZjZmZmBmYAZmZm
QAAAAAAiBmZgZmYGY2ZmZgZmBmZmZjAAAAAAIgdmYGZmZmNmVmYGZgZmZmYwAAAAAgYKZgAW
YmYmZmZmBmZmZhAAAAACBg5mAgYOZgIQAAACBQxmAARq6g5mAgAAAAIEDGYABquqRgAMZgIA
AAACAwpmAAhqpqqmDGYCAAAAAgMKZgAIcaaqowxmAgAAAAIDDGYABqasoQAMZgIAAAACAQpm
AAhnpqpgDGYCAAAAAgEKZgAGaaaqAA5mAgAAAAIACmYACGmlqrMMZgIAAAACAApmAAhsqKrC
DGYCAAAAAgAKZgAIeqaqoQxmAgAAAAIACmYACIqmqqAMZgIAAAACAApmAAg6pqqgDGYCAAAA
AgAKZgAIKqaqqQxmAgAAAAIACmYABhqtqgAOZgIAAAACAApmAAgWrKqmDGYCAAAAAgAMZgAG
qpqqAAxmAgAAAAIADGYACKqqqhYKZgIAAAACAAxmAAaqqqoADGYCAAAAAgEMZgAKqqqqsWgA
CGYCAAAAAgEMZgAKqqqqoGUACGYCAAAAAgMMZgAUqqqqoGZ2ZmZmEAAAAgMIZgAYdmaqqrYA
ZiZmZmYwAAACBAhmABh2ZiqqpmZmJmVmZkAAAAIFDGYAFGqqqmZmJmJmZmAAAAIGCGYADGdm
aKqqtgpmAnAAAAIGCGYADGJmZhqqpgpmAmAAAAAWB2ZmZmdiZmZgmqYACmYCYAAAABYGZmZm
aGZWZmYQBgAKZgJgAAAADgZmZmZoZiYAEmYCYAAAAA4GZmZmZ3ZiABJmAmAAAAIGCGYABiZo
JgAQZgJgAAACBh5mAmAAAAIGHmYCYQAAAgYeZgJiAAACBhZmAAo1ZmZmZAAAAAAMAAASNFZ2
CGYCZQwAAAAiAAAB
ENDofIMG
end

demo.show_image('bmp', '8-bit', <<'ENDofIMG')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==
ENDofIMG

if false
demo.show_image('bmp', '8-bit-RLE', <<'ENDofIMG')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ENDofIMG
end

demo.show_image('bmp', '32-bit', <<'ENDofIMG')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ENDofIMG

demo.show_image('gif', 'gif87a', <<'ENDofIMG')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ENDofIMG

demo.show_image('gif', 'gif89a', <<'ENDofIMG')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ENDofIMG

teapot = <<'ENDofIMG'
",
".. c #998074",
"#. c #84aadc",
"a. c #c47e55",
"b. c #4c6ea0",
"c. c #d48a64",
"d. c #ecccbc",
"e. c #8c6252",
"f. c #b1a094",
"g. c #714628",
"h. c #af8a76",
"i. c #2c5284",
"j. c #98745f",
"k. c #8f5124",
"l. c #844614",
"m. c #5074a6",
"n. c #aeb2b9",
"o. c #3e291c",
"p. c #44689b",
"q. c #4c2705",
"r. c #1b2b43",
"s. c #e1c8bb",
"t. c #dd782d",
"u. c #9c5824",
"v. c #547aac",
"w. c #906e5a",
"x. c #545253",
"y. c #e07e33",
"z. c #bcaeac",
"A. c #aeb6bc",
"B. c #6f737b",
"C. c #7e5942",
"D. c #b55e1a",
"E. c #7c5134",
"F. c #8c7a70",
"G. c #62524d",
"H. c #bc8a6c",
"I. c #936754",
"J. c #6e6d75",
"K. c #8cb0e2",
"L. c #d4beb4",
"M. c #2a201a",
"N. c #c48664",
"O. c #412105",
"P. c #a95a19",
"Q. c #747a7c",
"R. c #2a2e33",
"S. c #484c53",
"T. c #8690a1",
"U. c #6c524c",
"V. c #505967",
"W. c #040204",
"X. c #8cb6e4",
"Y. c #8f4912",
"Z. c #846e5c",
"0. c #244a7c",
"1. c #dcc2b6",
"2. c #33261b",
"3. c #81736c",
"4. c #3c3b46",
"5. c #735f53",
"6. c #545e69",
"7. c #bc7c58",
"8. c #4c2f1d",
"9. c #a4806b",
".# c #604c4c",
"## c #d4916b",
"a# c #5c80b2",
"b# c #e3d3cd",
"c# c #c8ad9a",
"d# c #bf6c2e",
"e# c #08121c",
"f# c #9c5e34",
"g# c #9298a1",
"h# c #8b7f7c",
"i# c #835f44",
"j# c #a46e54",
"k# c #474649",
"l# c #a65d26",
"m# c #be6420",
"n# c #7c9ed2",
"o# c #ac622c",
"p# c #726e6c",
"q# c #e1a684",
"r# c #a48676",
"s# c #54464c",
"t# c #a0a4ac",
"u# c #b37b5a",
"v# c #947a70",
"w# c #543d31",
"x# c #743e12",
"y# c #98bcee",
"z# c #d49e7c",
"A# c #64320c",
"B# c #d4cac6",
"C# c #2c3642",
"D# c #6e686b",
"E# c #5c463e",
"F# c #7f7575",
"G# c #bcb2b0",
"H# c #979dac",
"I# c #a47a61",
"J# c #8c9094",
"K# c #1c262c",
"L# c #d3c3bc",
"M# c #9cc2f4",
"N# c #af7455",
"O# c #fcad5f",
"P# c #6486b8",
"Q# c #d4742d",
"R# c #6d4e3d",
"S# c #242224",
"T# c #3b444c",
"U# c #181515",
"V# c #6e4024",
"W# c #749bcc",
"X# c #afabac",
"Y# c #7e6a64",
"Z# c #343634",
"0# c #e8ac8a",
"1# c #343a44",
"2# c #b18f7a",
"3# c #dda07f",
"4# c #6f380d",
"5# c #2f5684",
"6# c #a08d86",
"7# c #5f6068",
"8# c #625751",
"9# c #16253b",
".a c #94b6e9",
"#a c #dfcec8",
"aa c #3c6194",
"ba c #ecdad3",
"ca c #948a89",
"da c #a4c8fa",
"ea c #bcb6b8",
"fa c #604432",
"ga c #cc6d29",
"ha c #949ea4",
"ia c #413024",
"ja c #a48a7a",
"ka c #9a867d",
"la c #1c1a1c",
"ma c #342116",
"na c #bc8364",
"oa c #9c8674",
"pa c #844e25",
"qa c #3c4a5c",
"ra c #582c0d",
"sa c #c7835a",
"ta c #613c24",
"ua c #614b41",
"va c #adaeb5",
"wa c #805c4c",
"xa c #0a0b0f",
"ya c #c4b2ac",
"za c #648abc",
"Aa c #af846a",
"Ba c #c68a69",
"Ca c #483c39",
"Da c #7f400d",
"Ea c #646e74",
"Fa c #a85515",
"Ga c #042244",
"Ha c #816759",
"Ia c #95817c",
"Ja c #9c9aa1",
"Ka c #9c6b54",
"La c #7e4924",
"Ma c #0c1a2c",
"Na c #5f595f",
"Oa c #6f615e",
"Pa c #474149",
"Qa c #8c9aac",
"Ra c #cbbeb8",
"Sa c #806e68",
"Ta c #82624e",
"Ua c #949294",
"Va c #b4652c",
"Wa c #9c9ea4",
"Xa c #72574c",
"Ya c #2a303e",
"Za c #bc9175",
"0a c #9c7a64",
"1a c #261a13",
"2a c #aca6ac",
"3a c #341a04",
"4a c #593520",
"5a c #d49878",
"6a c #26282c",
"7a c #475261",
"8a c #3f3535",
"9a c #63676f",
".b c #a47255",
"#b c #bc967c",
"ab c #52433a",
"bb c #80a4d6",
"cb c #544c4c",
"db c #bc723c",
"eb c #645e54",
"fb c #a4abb4",
"gb c #c89070",
"hb c #8c7469",
"ib c #c7b8b3",
"                                                                                                                                ",
"                                                                                                                                ",
"                                                                                                                                ",
"                                                            wawae.e.wa                                                          ",
"                                                          .#j#N#7.a.db.#                                                        ",
"                                                            8.g.x#C.o.                                                          ",
"                                                  N#DaA#raY.Xa4aV#4aPas#q.W.W.6a                                                ",
"                                            i#O.l.4.4.4.4.4.s#wa7.N#wawaXawawawaA#q.C.                                          ",
"                                        3a3a4#wawawae.e.e.I.j#m#d#VaN#KaI.e.e.e.e.waDaO..#j#j#N#N#j#fa                          ",
"                                        O.A#wae.e.I.e.I.I.Kal#m#m#D.u.KaKaKaKaI.e.I.e.Da3aXaE#C#W.o.C.                          ",
"                                      w#e.k.I.I.Kaf#KaKaKaKal#m#m#D.l#j#Kaj#KaKaKaKaKaFai#u.    GaDaI.                          ",
"                                      .#waKaI.KaKaj#j#j#j#j#VaD.m#FaVaj#j#j#j#j#j#j#KaKae.U.    9#o.Ka                          ",
"                                      .#Xae.Kaj#j#N#N#N#N#N#j#N#j#N#N#N#N#N#dbdbN#N#N#I.waU.  e#M.taPa                          ",
"                                    .#U.Xawae.Kaj#N#7.dbj#a.a.a.a.a.a.a.j#7.sa##naj#I.e.waXaU.Yama.#                            ",
"                                    .#U.Xawae.I.Ka.bna3#0#3###sasaa.sac.##Ba7.u#N#Kaf#f#waXaU.4.s#Pa                            ",
"      j#N#5aW.Y.u#j#R#            8a.#XaXae.e.I.KaN#naz#0#q#gb7.7.N#N#N#u#nau#N#N#Kaf#f#e.waU.LaU.                              ",
"        4.Hahb.bI.j#KaY.          s#U.XaXawae.KaKaN#na3#0#3#gbna7.N#u#u#N#7.u#N#.bj#KaI.e.waXa.#                                ",
"              G.2#e.KaI.E#        .#U.Xawae.P.Kaj#N#H.z#q#3#gb7.7.7.7.N#N#u#u#N#.bj#KaI.e.TaXaU.0.                              ",
"                Ca.#e.j#Va  5#aa0.MaU.Xawak.d#I.j#I#N.5a3#5aBa7.dbN#N#dbN#N#u#I#j#KaKaI.e.waXaU.0.i.p.                          ",
"                  abU.I.KapaaaaaxaGaU.C.wau.VaKaj#I#na5a5agbN.u#N#7.N#N#N#N#N#.bj#KaKaI.e.waXa.#R#m.m.m.p.                      ",
"            zaW#0.0.s#waKam#0.0.e#GaU.C.k.l#I.KaKa.bnagbgbBa7.u#7.N#N#N#N#N#.bj#j#KaKaI.e.waXa.#.#b.m.m.i.i.0.                  ",
"    5#aaW#W#n#aaaa0.U..#e.Kau.ta4aLak.k.u.o#e.KaKa.bu#H.H.nau#N#N#N#N#N#N#j#N#j#KaKaI.I.e.waXa.#fab.m.i.i.i.i.i.0.              ",
"p.p.p.aan#W#0.0.0.0.p.uaU.KaKal#o#d#gal#o#o#e.I.Kaj#N#Aau#u#N#N#N#.bN#N#j#.bj#j#KaKaI.I.e.waXa.#E#m.m.v.m.i.i.m.n#bbn#          ",
"p.p.n#n#n#p.0.0.0.0.0.xa.#waKaKagaQ#t.y.d#dbi#e.I.Ka.bN#I#N#N#N#N#.bN#N#j#N#j#KaKaI.f#e.waXaU.U.w#5#m.m.i.i.i.m.bbbbbbbbaa      ",
"n#n#n#n#n#n#0.aaaap.aaaas#s#waKaj#na##O#dbl#wae.I.I.j.j#j#j#j#j#j#j#.bj#j#j#KaKaKaI.e.e.waXaE.Hai.m.m.i.i.i.i.p.p.bbb.b.p.p.aa  ",
"p.bbn#bbp.p.p.p.p.p.aaaaaaua..#be.Kaj#j#u.XawaTae.I.KaKaKaKaj#j#j#j#j#KaKaKaKaKaI.I.i#waXawaR#uav.m.m.v.v.i.b.bbbbbbbbp.p.b.p.p.",
"bbp.p.p.p.p.p.n#p.0.0.0.0.p.8#Oa2#ZaZaZak.XaXawae.e.e.I.KaKaKaj#KaKaKaKaI.w.I.I.I.e.waXa0aXa5.5#i.m.m.v.v.p.p.#.#.#.#.#.#.b.bb#.",
"bbW#p.p.p.n#bbbbp.0.0.0.0.0.0.0.x.D#3.3.w.waU.wawae.e.e.e.f#I.I.I.I.I.I.I.f#e.e.wawaXae.w.ua4.5#5#i.7ai.b.b.b.p.b.#.#.#.p.b.b.#.",
"p.bbp.bbbbp.p.p.p.0.0.6.p.p.p.0.0.0.0.9#R#wae.9.j.wawae.Tae.e.e.e.I.e.Tai#e.wawaw.Aaw.hbOa6aC#5#m.v.v.#.p.p.b.b.p.#.#.b.b.p.p.b.",
"0.0.0.p.p.p.p.aa0.0.p.p.p.p.p.p.0.0.p.qaT#s#Z.0a..h.I#I#ZaZaI#gbI#.bH..b.bH.h.j.r#HaU.PaV.V.Na1#v.m.K.K.K.#.b.#.#.#.#.K.p.b.m.p.",
"0.0.0.0.p.0.0.0.0.0.0.p.p.p.p.0.0.p.p.qaqaqa2.iaR#HaY#ja2#2#I#Za2#0a2#2#h.jaoaF.8#V.6aC#C#C#CaNab.bbbbK.#.b.b.b.#.K.K.#.#.#.X.#.",
"b.b.p.p.p.0.i.0.0.0.0.p.0.0.0.i.0.0.qaqak#6aR.6a3aiawaY#OaOaOaD#Iah#D#OaF#cbPa6a6aW.V.6.1#C#7ap#b.b.b.b.b.b.b.b.b.K.K.#.b.b.b.b.",
"bb#.#.#.#.b.b.i.0.p.p.p.0.b.b.0.0.0.qar.YaYaYaR.K#Z#E.xa1a2.S#.#x.x.S.R.laU#1#W.W.9a6.J#J#T.7av#K.K.K.K.b.b.m.m.b.m.K.b.p.m.b.b.",
"#.#.#.#.#.#.K.#.#.#.#.#.#.b.b.b.#.p.qaqaYaR.YaR.Pa6ak#S#lalaW.W.xaW.W.W.W.U#U#ebT.hag#g#haT.7ac#K.K.K.K.K.K.b.b.#.K.K.K.K.m.b.b.",
"#.#.#.b.b.b.b.#.#.bb#.#.#.b.K.bbK.#.T.T.J#R.YaS.S.x.ebebR.2.S#S#S#T#S#S#S.9a7aEaEafbhag#6.7aV.m.K.K.K.K.X.K.m.X.K.K..a.aX.K.b.m.",
"b.b.p.b.p.b.p.K.K.K.b.b.b.b.#.#.#.#.T.T.J#UaJ#J#g#g#haWaWa7#9a2at#fbA.t#9a6.9aJ.EaEafbV.6.V.Nam.b.m..a.am.b.m.m.m.K.X.K..a.a.am.",
"  p.b.p.p.b.p.K.b.b.p.b.b.b.b.K.#.K.7a7aV.9aWat#X#WahaH#ha6.t#n.n.n.n.fbfb9a9an.t#t#fbfb7aV.7#m.m.m..am.m.m.m.m.m.m..a.a.am.m.m.",
"  b.p.b.#.#.#.#.#.b.b.b.b.b.b.K.Pax.V.7aNa7#7#t#vafbB.B.B.B.B.n.A.A.A.fbfb9aA.t#t#fbfbQaH#7#D#m.X..a.a.aX.v.v.v.v.m.m..am.v.m.m.",
"    #.K.K.K.K.K.#.b.b.b.b.K.K.K.T.S.V.9aB.ebNaV.p#9a9aB.D#9a9aB.A.A.A.9aB.B.Eafbn.t#g#hat#X#m..ay#y#y#y#.ay#v.v.v..aX.y#.av.v.v.",
"    #.K.K.#.K.K.K.K.K.K.K.K.K.QaT.V.7#Ea9af.f.2at#OaD#F#Q.J.9aEaB.Ea9a6.Ea9aEaV.H#H#t#2aD#J.v.m.y#y#y#y#X..ay#y#y#y#y#y#.ay#.av.",
"      K.K.K.K.b.b.b.b.K.K.K.K.g#g#t#D#fbWag#c#X#2a2aeb7#D#Q.vaA.2at#p#9a6.V.7#7#9avaSaSaSav.m.v.v.y#.ay#X.v.v.v.v.y#y#y#y#y#y#y#",
"      K.K.b.m.b.b.b.b.K.K.K.K.Ja7#7#6.2az.#ac#z.G#eaG#p#2a2a2at#t#t#t#7#9aD#J.SaL#RaRaF#v.v.v.v.v.v.y#m.v.v.a#a#v.v.M#M#M#y#y#  ",
"        b.b.m.b.m.b.b.b.K.X.7#7#7#7#D#3.1.z.yac#yaibh#caB#L#eaz.z.z.z.z.F#h##ab##aL#ibibv.a#m.m.v.y#M#y#a#v.v.v.a#a#a#a#M#      ",
"        p.b.m.b.b.m.b.b.m.b.7#9a9aSahb..m..ayayahbhbF.h#Ias.RaRaibibL.L#kab#bab#1.L.L.y#y#a#v.v.M#y#y#M#y#v.a#a#a#v.a#a#        ",
"          b.m.b.b.X.K..aK.b.7#D#F#hbv#m.b.m.m.F.F.F.v#v#F.v#L.L.L.L#..ka6#h.s.1.L#1.y#M#y#daM#M#day#y#M#M#M#a#a#a#a#            ",
"          p.b.X.X.X.X.X.X.X.9aD#6#..m..aX.y#m.m.F.v#v#v#....1.Ia....kakar#r#r#1.s.M#M#y#a#a#a#M#M#y#y#daM#dadada                ",
"            .a.aX..aX..aX..aD#SaF..ay#y#.aX..am.m.........s.s.9.9.r#r#r#r#kakar#M#M#a#a#a#a#a#a#M#M#M#M#dadada                  ",
"            K.X..a.a.a.a.aH#vab#1.y#X.X.y#y#y#y#v.v...s.d.d.d.s.r#r#h.jajajar#a#M#v.a#a#a#a#a#a#P#dadadaM#                      "};
ENDofIMG

demo.show_image('xpm', 'color', "/* XPM */
static char * teapot[] = {
\"64 48 204 2\",
\"   c #145ec4#{teapot}")

demo.show_image('xpm', 'transparent', "/* XPM */
static char * teapot[] = {
\"64 48 204 2\",
\"   s None c None#{teapot}")

demo.show_image('xbm', 'bitmap', <<'ENDofIMG')
#define flagup_width 48
#define flagup_height 48
static char flagup_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x7f, 0x00,
   0x00, 0x00, 0x00, 0xe0, 0x7f, 0x00, 0x00, 0x00, 0x00, 0xef, 0x6a, 0x00,
   0x00, 0x00, 0xc0, 0x7b, 0x75, 0x00, 0x00, 0x00, 0xe0, 0xe0, 0x6a, 0x00,
   0x00, 0x00, 0x30, 0x60, 0x75, 0x00, 0x00, 0x00, 0x18, 0xe0, 0x7f, 0x00,
   0x00, 0x00, 0x0c, 0xe0, 0x7f, 0x00, 0x00, 0x00, 0x06, 0xe0, 0x04, 0x00,
   0x00, 0x00, 0x03, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x01, 0xe0, 0x06, 0x00,
   0x00, 0xc0, 0x1f, 0xe0, 0x07, 0x00, 0x00, 0xe0, 0x7f, 0xe0, 0x07, 0x00,
   0x00, 0x70, 0xe0, 0xe0, 0x05, 0x00, 0x00, 0x38, 0x80, 0xe1, 0x04, 0x00,
   0x00, 0x18, 0x80, 0xf1, 0x04, 0x00, 0x00, 0x0c, 0x00, 0xfb, 0x04, 0x00,
   0x00, 0x0c, 0x00, 0xff, 0x04, 0x00, 0x00, 0x86, 0x1f, 0xee, 0x04, 0x00,
   0x00, 0x06, 0x06, 0xe6, 0x04, 0x00, 0x00, 0x06, 0x00, 0xe6, 0x04, 0x00,
   0x00, 0x06, 0x00, 0xe6, 0x04, 0x00, 0x00, 0x06, 0x00, 0x66, 0x04, 0x00,
   0x7f, 0x56, 0x52, 0x06, 0xe4, 0xff, 0x00, 0x76, 0x55, 0x06, 0x04, 0x00,
   0x00, 0x56, 0x57, 0x06, 0x04, 0x00, 0x00, 0x56, 0x55, 0x06, 0x06, 0x00,
   0x00, 0x56, 0xd5, 0x06, 0x03, 0x00, 0x00, 0x06, 0x00, 0x86, 0x01, 0x00,
   0x54, 0x06, 0x00, 0xc6, 0x54, 0x55, 0xaa, 0x06, 0x00, 0x66, 0xaa, 0x2a,
   0x54, 0x06, 0x00, 0x36, 0x55, 0x55, 0xaa, 0x06, 0x00, 0xbe, 0xaa, 0x2a,
   0x54, 0xfe, 0xff, 0x6f, 0x55, 0x55, 0xaa, 0xfc, 0xff, 0xa7, 0xaa, 0x2a,
   0x54, 0x01, 0x88, 0x60, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa0, 0xaa, 0x2a,
   0x54, 0x55, 0x8d, 0x60, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa0, 0xaa, 0x2a,
   0x54, 0x55, 0x8d, 0x60, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa0, 0xaa, 0x2a,
   0x54, 0x55, 0x8d, 0x50, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa8, 0xaa, 0x2a,
   0x54, 0x55, 0x95, 0x54, 0x55, 0x55, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x2a,
   0x54, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
ENDofIMG

demo.show_image('png', 'color', <<'ENDofIMG')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ENDofIMG

demo.show_image('png', 'grayscale', <<'ENDofIMG')
iVBORw0KGgoAAAANSUhEUgAAAC4AAAAjCAAAAAG5S1ejAAAEJElEQVR4nKWTXUwcZRiFH2YG
ZpiBLsx2wW6ZpU1YC+3axWClf7ZKY5NVozHRxIYaL0xq1YuGeOFVRW40TXthk4aLxsT/xJjG
+tdiGkUl1nQV6rYokuy2IFugy8/irp3dHXYGvFiwBdQ08Vx9+d73Oznfe84LAIjQNv+ELdBy
aVqTRBSh/8p6aGMZWpDomBwwETg7020sL0NjG0XsTTT9GmlSdreji1xZ555bvzmyVvcNAI/s
pRX/8lfi4uHoREIEnrnEKwNjcxUibdt+39PX6gwNTEkkB0l+3VPtdgclQI8fs0zd8gvoMkZX
MBywY3B0Ox0FKoGcTvtKhQuQa5tBuvXmkHOCx4Y1xS7o6Qg29vFi6DvMe79paRjP5sIgAL/1
jxwKJC4ewS72+kp7WfX4UQRgFUlBjA1/2R7t25DQdpbZiQJPU5V1xVMuqW+9ykc+X37WKJLC
AmD8gkJX0HIOS3S3f+B3h48jAVNI0Viztbub4wCdC+rUf/3J7eEBOgr6l0Dbr4TUpffBl0Jw
cARr5qYvL2wPM1d8bv65a3kzPTKKBHU7AuFT53eAS3FaSsdyynQcRGiNxXaVff9wfa+Of/rB
T66OpgkmRNiYkkssj6CuDusJ6s4YDTUHTjsCuEiNauQu7MNqXJ8LekGoRwAVXNGkV+1/33Wn
mVdERYx5EWDCttbJqpQ5NqCRSgV0UbE0BBjzP/XpBDn90Te3ll3q8aXXqGIVAvSInau8rrSV
m5iaY9DaU6TUdyKAKVUhWh/qSU6Owmv9mv9wIQ+OxynqbpZ6IQ6cAgrzURXtc/ZoDnQtGWFH
6z/OW/t/dhkARbfR2DGer5avbnyZZWleCv/+C13AkYmP3bkxqy73X+y6v9F9NgLNW8Sf7DJs
yzI+X8ne3DK99r0YoDNSEUHbdb95RiozLWR5TM2ABP79fyRlc8gfiKZ7wuFQ5dOR01CU8vpp
kZMp05St+UFrga4IQN3npMg7W2/8UCm5k9uT2TcgID7/rqkF/LnDANTvkNYNvlMIZn6L73re
YfWMEdEynm8bSkIXs3Lppki21G3fsGpr3bU1yvS21InFHO8sjTMfryzNXIxbmyPXhPrm0Bkl
fXdGKr/DfVl2ZEEQqJhl9O91SF+fFMtWR+8KeKdsogdSxe3ynCdnR0dn3IqmuFyKkqwxFtmN
alutu2/qx3FRsZuGXFm7a8OsL+r5ikSFx53Vb5RJomBWO30L7WaDNTN8fthTrLrmZHnGVz58
4Z4ST81nU3/Khl6SnbWy5ZVGxu5btOmQfa7GUmRLzZhmU0oyu8B4SOwEEJ/VsFJVCq/ftCm/
xnDZOVC+gCd/dnqB+NsH1QzgnLzFxoX2CoGELiu9kxBKpo1JAOv4ygwvhsDcNC6lTgF7h25u
g7mivaD9QKlyOZxcUVyJvwDv65E2xA9RYAAAAABJRU5ErkJggg==
ENDofIMG

demo.show_image('png', 'transparent color', <<'ENDofIMG')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ENDofIMG

demo.show_image('png', 'transparent grayscale', <<'ENDofIMG')
iVBORw0KGgoAAAANSUhEUgAAAC4AAAAjCAQAAAE2KcD0AAAFPElEQVR4nMXWXYicZxUH8N+z
M3HezCy76axJzJrtR8jWVVc29IMtTTG6hUqqqL1QWlrxQmirXpTihRdSY4pUSws1IIEWBesH
CAZFapNaSISgpUurJra2i7u2IWs+dpNMnbWzndfM9Hixs5tNmrYJCv7vHvi///ec85z/OQ/L
kODeeMZ1fq+LsTjolIqixAejx3jqj5EFknfEWFBke5zwkga62O01+9LAO38INsW9QeKmmHG1
vzrgapkttiWqkWA0hsziSr/1CX+T25fAp+Km4PZg8F2CfCseipGga+HwxYBvxur4hWQoigtp
T7grHk3rbXHAVKJIzQRq9sZ+a/XpixHFBZGqaQ/LNVTlBnVRVcKAPWnEeBrWMrXw7+uD7WfF
3EVTFdvSxaaiFJfFaGAxlLfinmj7fuIzcUhFpmUZfXvUND2a+Gr0uT+x1+cxFhsc1zDvQOrE
CC97wWH3xHDM+JMHg5YV+uNSKz2PHrfEQ2eK24OaLgVTDnnKtpj0Rx8wo+IG3VpmLFNnSNW4
futUZb7tW9b6Xcq1rFe2QdU/ztAHvIgMe9KIXDvdl4rYl7aln6VBfcbtSEupnkTRpKk0Grkt
9mHHUsV3nl378kV31f8SH1/qkbe91UVU4hanbI0ylhXybIzE12JrwN0OI/caOqPgDL4SK+xI
rI5rPK2d7opZdU3zSsYXC7kxNhs2blf6g82gV6adxmKlo5oyp0yf6Zmtpu1xrc/Gi9a6O2DG
WJyy0SsOpvE0nRiJjvq/UHJUZovDun0ynkyVOKak5kQaiH7c7IFF9V7UHVFB07NuQ26TKzSN
RH+nJkOL9HIn3kk1/cpe8NPodaWG0zIFmYIp/Yv0WS25y5WUFc172EsqqKsbVlWQyVUW6UcN
utWvzVow9Kf9wHW6HUz7XWrOOmUFaxbp+xXs1KNfrzm5ptl00puYSLkbJZkhO1OH3khFa1CQ
+7mqGh5LR8AD6QUVg+5Ly3qmbbW2ZF8ajaLnwXTnpnctu/lOz5RlKp5I3KiinWDP2w+k7XH7
hVuk8v9008DSz9/VTBeG7XHMaWuVvOJDvv7fiA/GHZ5dKvODMeuX+jQdlduoucS7yMVQjUGb
9Nndmd6jca2C57R0oyWXG/BEuoDIR2PMKe/3E1MdehWHrXIgUYmP+piGJxV1a8hRUnJUOebP
NONg3OGfakoaXjVo2KQ5+42ncVvjEl9wIH6VFhKt6zeIsSipqWtoKMmFCfk5dVg6luM2bXWc
1nad1z3jEkV9aq5X84bvJRiOgi/7sYaKYYOaHcssYCg2K7rchMfTslk6nwqu6EjTVHeVQ3Iz
qnbJ9PhuVIM65v1bjrbjTtoUo3F9bIrRGI1eL9tgzuPL9ymsxGkwbVbJvBnjpqxSs9srqr7h
kaiZt9cN+mVaeqwXCtrKCgoKOG7ewlPgnC0w57gTCrq916SPGNbvpBYm3ZnqVtim5E2rNbVM
OuI1fTIVmV69Mpma9RYedcvEj2DAgKvcqmmvY15X1LJBr4qW1XF/+rtTblRTM2HWIbMY0WMd
Mpmyklnz+s4Vf05TpumgH9ons1q3XiUzNmq6wjAeSU3vcY33qaWJdDS96g1Vl0lKGuoyawyp
ndstcE+0PG29XKYkVzavoeFqdUWNjicH4maFZW+DQnxJBbm6NTJ853wmOm2dAb1aHQtnfpPg
c/Fn7c4oZjr9KO5eMgrt9Jjz4yzxVbowo6ok87wTCbZGzZyBzgnytOOCBu859m/4sGOK6ktL
4qZ49byboHEBU2kZ5c5YKfMX42oX/849L/4D4jD0vGNZ6YMAAAAASUVORK5CYII=
ENDofIMG

demo.show_image('jpeg', 'color', <<'ENDofIMG')
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/2wBDAQkJCQwLDBgNDRgyIRwh
MjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjL/wAAR
CAAmADkDASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAA
AgEDAwIEAwUFBAQAAAF9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkK
FhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWG
h4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl
5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBAQEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREA
AgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMzUvAVYnLRChYk
NOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOE
hYaHiImKkpOUlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk
5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwDzLyJSuV5FIsUm4DAzW1Ywpb3sbPhoQ4LA
jPGea6jWdH0W6EdxYLJA78gICUb169KIq8JTutDWnSnUmqcFds4y2e5jkzGrOVGTt7Ct2aae
ZFzGSR6cmni1t4RiFWCseRknJHf+f51MsZG4JHwP4hXmVaqctEfVYDLqsKbTla+43S7NZLkr
dHamMEgg9R6isDVtDu7KdjFGZoOodOcD3rowWIVRkbDuUHI3DFbWnOs1n9pIIMZwwIzyBxwe
v/16ujWs9iK+VQqfFN3W3Yo6Pp/n6LbXFx9rWeCNifLbjac9T6YxXN+a/wDzy/SvedC13S5/
DtnZ3ckZmMQSVVTK8cYJ6HI96tfZvC3/AD42/wD35NdfM5dT5Pdtp3PBtEji1CVTtyR1U9vc
11WpXOnRW9rYwWxB6O+/n2OOgrm40EZyg2n2rTm8M6xp+lrrk1q76bIu4hcMwB74zkD3/TFV
KcXRcFHXudWGm6deM+axUmCQytGjrIi8Kyg81NFKrxqogVsMWk5OH78j8/YfnQ1hKlrbXU8F
xCk43ws67d4/L3H5j1rLurqVLhhno2RXlKErn2f1qEIKV7xL77QN21VxwMdqdLdvBpbpDveS
5cJGiHBzyMj8/wBB61i3OoySJjhR3C0+71FYYfIjncSgbSUOMj0J/mBW1OlJSuzz8fmsY0nG
kvQ6qO4ubaLyt8fnR/Idh3AEdfyNQeVrH/QQuf8Avv8A+tWd4fu0j0dePnEjIM+vU/zqz/aq
f8/J/wC/h/xotNSaifEO6bsVo1yy8DmvetkF3HbaW0INuYFMiYwpXA4wKKK7Y7HWtzSvNPs7
ywezurdJLYrjyyOgx29D6EcivGviF4EtfDlnFqVpeSvFLMsPkSKCVJVjncO3y9MfjRRRUirX
NKdWcfdi9DhbW3YRS3a7cW+GOepycDA+vWsaS3AcqScjvRRUU9zOcnLVlqx8+Elll43qoH1r
K2L70UVaerMluf/Z
ENDofIMG

demo.show_image('jpeg', 'grayscale', <<'ENDofIMG')
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/wAALCAAmADkBAREA/8QAHwAA
AQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAAF9AQIDAAQR
BRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RF
RkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ip
qrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/9oACAEB
AAA/APMvIlK5XkUixSbgMDNX7Z7mOTMas5UZO3sK3Zpp5kXMZJHpyan0uzWS5K3R2pjBIIPU
eorA1bQ7uynYxRmaDqHTnA966/R9P8/Rba4uPtazwRsT5bcbTnqfTGK5vzX/AOeX6VFYwpb3
sbPhoQ4LAjPGea6jWdH0W6EdxYLJA78gICUb169KzBa28IxCrBWPIyTkjv8Az/OpljI3BI+B
/EKcCxCqMjYdyg5G4Yra051ms/tJBBjOGBGeQOOD1/8Ar16PoWu6XP4ds7O7kjMxiCSqqZXj
jBPQ5HvVr7N4W/58bf8A78mvBtEji1CVTtyR1U9vc11WpXOnRW9rYwWxB6O+/n2OOgrDmCQy
tGjrIi8Kyg81NFKrxqogVsMWk5OH78j8/YfnUT7QN21VxwMdqdLdvBpbpDveS5cJGiHBzyMj
8/0HrWnHcXNtF5W+Pzo/kOw7gCOv5GoPK1j/AKCFz/33/wDWrIjQRnKDafatObwzrGn6WuuT
Wrvpsi7iFwzAHvjOQPf9MVE1hKlrbXU8FxCk43ws67d4/L3H5j1rLurqVLhhno2RVa51GSRM
cKO4Wn3eorDD5Ec7iUDaShxkehP8wK0vD92kejrx84kZBn16n+dWf7VT/n5P/fw/41WjXLLw
Oa962QXcdtpbQg25gUyJjClcDjArSvNPs7ywezurdJLYrjyyOgx29D6EcivGviF4EtfDlnFq
VpeSvFLMsPkSKCVJVjncO3y9MfjXC2tuwilu124t8Mc9Tk4GB9etY0luA5Uk5HerVj58JLLL
xvVQPrWVsX3r/9k=
ENDofIMG

demo.show_image('jpeg', 'progressive color', <<'ENDofIMG')
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/2wBDAQkJCQwLDBgNDRgyIRwh
MjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjL/wgAR
CAAmADkDASIAAhEBAxEB/8QAGgAAAgMBAQAAAAAAAAAAAAAAAAQDBQYBAv/EABkBAAIDAQAA
AAAAAAAAAAAAAAECAAMFBP/aAAwDAQACEAMQAAABzA7aRqV/vvl1eIWLrCCt3stuTgrWtaNs
PsV5dljqXX57XwtLMSLehdazjQjUSQBZVQK//8QAHhAAAgIDAQEBAQAAAAAAAAAAAQIAAwQR
EiEQBRP/2gAIAQEAAQUC0ZoxegSSZUm2toZDTXuncQcvdTS05UTXyv1KL6jj84spAsNjVhT4
QfIW1UGZRq6DyHGurq4PLMemsJjWaGO2qf6wTxg9aOn6GCuOir4VibHz/8QAHxEAAgEEAgMA
AAAAAAAAAAAAAQIAAwQQERIxEyFB/9oACAEDAQE/AVUseIlC3YL3HtQ3ZxTPFwdzygDfyV7o
BdLlWI9CE7x//8QAGxEAAwEBAAMAAAAAAAAAAAAAAAECESEQEjH/2gAIAQIBAT8BXxsqukWb
o2vTMMZMvTu8EUifH//EACgQAAEDAwIDCQAAAAAAAAAAAAIAAREQITESUQMiMhMjM0FScYGR
of/aAAgBAQAGPwKllhcys0shItUs3ksJtkxBLKyxTVsgAnaYuugfqggwqMrHvS2SwotLLxCp
27j3boSdiacVhnuvldX7QeFHLF1oIZHZNxBN4d4h05emmaf/xAAgEAEAAwACAgIDAAAAAAAA
AAABABEhMUFRcWGBEJHx/9oACAEBAAE/IcshZFgLXiHkBRkjX2ZMMXcupaDQ7fZRlXxwzq2m
HKvtNw8afMR+4nAyQDPwUhn1PcOblWCA4SAgq7fyjXNBFYupQJoHCzZ/Qg6ZDC0rzWJF3bSr
hl7gHj1CvvTIIebEpAaSnHFp1UQioh8wNbjHv6jUa7TSpzzsEon/2gAMAwEAAgADAAAAEIb7
MuvRaP43Yv/EABoRAQADAQEBAAAAAAAAAAAAAAEAETEhUXH/2gAIAQMBAT8QGjax8NL2bzZn
k1Ubj8VKe1w5H5Gxahs6xyN0w2f/xAAaEQEBAQEBAQEAAAAAAAAAAAABABEhMUFR/9oACAEC
AQE/EBu05G8Fg+WvqUj1+xtC02FE8QZt6h63/8QAJBABAAICAgIBBAMAAAAAAAAAAQARITFB
UWGRcaGx0fCB4fH/2gAIAQEAAT8QvtkSkKLl8RFvQllJTrLEKYaUR2dkTm2mejzMlLHRa9vV
VP3kopAoS8XmEZJwK929QBQEyWtpz9/cQpiOEFAWXobKVLxEVQS8hjDv+46IjecMUumzzP8A
cRqyTa48sXGOt/w1ogxTIDmGiosjXPJ78HuVGgwVxK9tt03ks9/Q7nZ8b0E36Z+z/EFWbeJY
AwKAPNXYefpUHWepPryezuBl6sSs4HJFYsll1Z0v3CYYxDfe37zz/b8yzAZldCmppUMURDFp
Y0Vx09JklKrxFSl046V/MBjlFe22ij53AGls5lpwah8z5p//2Q==
ENDofIMG

demo.show_image('jpeg', 'progressive grayscale', <<'ENDofIMG')
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/wgALCAAmADkBAREA/8QAGgAA
AgMBAQAAAAAAAAAAAAAAAgQDBQYBAP/aAAgBAQAAAAHMenfNC3rRtFT67pJcFaomPWQTaBWP
rMsW9ZxtElKr/8QAHhAAAgIDAQEBAQAAAAAAAAAAAQIAAwQREiEQBRP/2gAIAQEAAQUC0Zox
egSSZUm2toZDTXuncQcvdTS05UTXyv1KL6jj84spAsNjVhT4QfIW1UGZRq6DyHGurq4PLMem
sJjWaGO2qf6wTxg9aOn6GCuOir4VibHz/8QAKBAAAQMDAgMJAAAAAAAAAAAAAgABERAhMRJR
AyIyEyMzQVJxgZGh/9oACAEBAAY/AqWWFzKzSyEi1SzeSwm2TEEsrLFNWyACdpi66B+qCDCo
yse9LZLCi0svEKnbuPduhJ2JpxWGe6+V1ftB4UcsXWghkdk3EE3h3iHTl6aZp//EACAQAQAD
AAICAgMAAAAAAAAAAAEAESExQVFxYYEQkfH/2gAIAQEAAT8hyyFkWAteIeQFGSNfZkwxdy6l
oNDt9lGVfHDOraYcq+03Dxp8xH7icDJAM/BSGfU9w5uVYIDhICCrt/KNc0EVi6lAmgcLNn9C
DpkMLSvNYkXdtKuGXuAePUK+9Mgh5sSkBpKccWnVRCKiHzA1uMe/qNRrtNKnPOwSif/aAAgB
AQAAABCrPPK25v/EACQQAQACAgICAQQDAAAAAAAAAAEAESExQVFhkXGhsdHwgeHx/9oACAEB
AAE/EL7ZEpCi5fERb0JZSU6yxCmGlEdnZE5tpno8zJSx0Wvb1VT95KKQKEvF5hGScCvdvUAU
BMlrac/f3EKYjhBQFl6GylS8RFUEvIYw7/uOiI3nDFLps8z/AHEask2uPLFxjrf8NaIMUyA5
hoqLI1zye/B7lRoMFcSvbbdN5LPf0O52fG9BN+mfs/xBVm3iWAMCgDzV2Hn6VB1nqT68ns7g
ZerErOByRWLJZdWdL9wmGMQ33t+88/2/MswGZXQpqaVDFEQxaWNFcdPSZJSq8RUpdOOlfzAY
5RXttoo+dwBpbOZacGofM+af/9k=
ENDofIMG

demo.show_image('tiff', 'uncompressed', <<'ENDofIMG')
TU0AKgAAHggAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAADb///bttu2tv+Sttu2ttvb2/8AJEkAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAD////b//+229u2
29vb2/+2ttttttu2ttu22//b//8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAJAC229skbZLb//+2tv8AJEkkbZJtkrbb///b//9tktvb//8A
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACSttu2
29u2ttvb2/8AAAC229skbZIAACSSttu229uSttsAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACQAAAAAAAAAAAAAJCQAJCSSttsAAACS
ttvb//8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAASUkAAABJbZIAACQkAACStrYASQDb2/8AAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAD/////29v//9v/29v///8AAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AABtJACSbSS2bUnbkm22km3btm22kkn/25L//9ttJAAkAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACSSSSSSSTbkm2SSSSSSSSSJACSSSS2
bUm2bUm2bUn/27b//9v/////////////27b/tm3btm3//9v/27b//////////////9skAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAkAABJAAAAAACSSSRJAABtJABtJACSbSSSSSTbtm22km3btm3btm3btm22km3/
km22bUn/25L/27b/25Lbtm3btpLbkm3/27bbtpLbtm22km3/25L/27b//7b/27b//////9v/
/////9v/////tpL/km1tJAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABtAABJAABJAABJAACSJCRJ
AABtJABtAACSSSRtJACSbSSSSSTbkm222222km22bUm2kknbtpLb25K2km22kknbtpLbtm3b
tpLbtm22km3btpK2km3btpLb25L//9v/25L/25Lb29v//7b/27b//7b/27b/25Lb29v/27bb
tpL//9v//////7b///////////////+2bUkAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAABJAABJAABJAABJAABJJACSSSSSJABtSQCSSSSSSSSSSSSSJCSSSSS2
bUmSSSSSJABtJABtAACSSSS2kkm2bUnbkkm2kkn/25K2kknbtm22km3bkm22km3/tm3btm3b
tm3btpLbkm22km22kkm2kknbtm3bkpL/25L/27b//7b/27b//7b/////tm3btpL/27b/tm3b
tpLbtm3//7b/25L//9v/////////////////////2/8AAAAAAAAAAAAAAACSSSQkAABtJABJ
AABtJABJAABJAABJAACSSSRtSQBtJABtSSS2kkmSJCSSJCSSSSSSSSSSSSSSSSSSSSSSSSRt
JACSSSSSSSS2bUm2km3//9u2bUm2bUm2bUm2SSS2bUm2bUm2km2SSSSSSSTbtm3btpL/25K2
km3btkm2km3btm22km3bkm3btpL/25K2tm3bkm22km3btm22bUm2bUm2bUm2kkm2bUnbtm22
tm3bkm3b25L//9v///////////8ADgEAAAMAAAABAEAAAAEBAAMAAAABACgAAAECAAMAAAAD
AAAetgEDAAMAAAABAAEAAAEGAAMAAAABAAIAAAERAAQAAAABAAAACAESAAMAAAABAAEAAAEV
AAMAAAABAAMAAAEWAAMAAAABACgAAAEXAAQAAAABAAAeAAEaAAUAAAABAAAevAEbAAUAAAAB
AAAexAEcAAMAAAABAAEAAAEoAAMAAAABAAIAAAAAAAAACAAIAAgSwAAAAAQAABLAAAAABAAA
ENDofIMG

demo.show_image('tiff', 'packbits compressed', <<'ENDofIMG')
TU0AKgAABAqBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEA
wQCBAMEAgQDBAIEAwQCbAAvb///bttu2tv+Sttv/tv7bA/8AJEm8AKEA/v8G2///ttvbtv3b
Df+2ttttttu2ttu22//b//+/AKMACCQAttvbJG2S2////7YR/wAkSSRtkm2Sttv//9v//22S
/9v//8IAoQADkrbbtv/b/7b+2wD//gARttvbJG2SAAAkkrbbttvbkrbbwgCcAAAk9wD/JAUA
JCSSttv+AAGStv/b///CAJcA/0n+AAJJbZL/AP8k/wAIkra2AEkA29v/wgCBAMEAgQDBAIEA
wQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQD9///b//8B2///2/7/gQDQ
AB5tJACSbSS2bUnbkm22km3btm22kkn/25L//9ttJAAkgQDgACOSSSSSSSTbkm2SSSSSSSSS
JACSSSS2bUm2bUm2bUn/27b//9v3/w3btv+2bdu2bf//2//btvb/AdskigADJAAASfwAU5JJ
JEkAAG0kAG0kAJJtJJJJJNu2bbaSbdu2bdu2bdu2bbaSbf+SbbZtSf/bkv/btv/bktu2bdu2
ktuSbf/bttu2ktu2bbaSbf/bkv/btv//tv/btvz/ANv8/wDb/f8GtpL/km1tJLUAYm0AAEkA
AEkAAEkAAJIkJEkAAG0kAG0AAJJJJG0kAJJtJJJJJNuSbbbbbbaSbbZtSbaSSdu2ktvbkraS
bbaSSdu2ktu2bdu2ktu2bbaSbdu2kraSbdu2ktvbkv//2//bkv/bkv7b//8Mtv/btv//tv/b
tv/bkv7bCP/bttu2kv//2/z/ALb1/wK2bUnXAH9JAABJAABJAABJAABJJACSSSSSJABtSQCS
SSSSSSSSSSSSJCSSSSS2bUmSSSSSJABtJABtAACSSSS2kkm2bUnbkkm2kkn/25K2kknbtm22
km3bkm22km3/tm3btm3btm3btpLbkm22km22kkm2kknbtm3bkpL/25L/27b//7b/2wO2//+2
/f8Ztm3btpL/27b/tm3btpLbtm3//7b/25L//9vx/wHb//UAAZJJ/yT/AH9tJABJAABtJABJ
AABJAABJAACSSSRtSQBtJABtSSS2kkmSJCSSJCSSSSSSSSSSSSSSSSSSSSSSSSRtJACSSSSS
SSS2bUm2km3//9u2bUm2bUm2bUm2SSS2bUm2bUm2km2SSSSSSSTbtm3btpL/25K2km3btkm2
km3btm22km3bkjBt27aS/9uStrZt25JttpJt27Zttm1Jtm1Jtm1JtpJJtm1J27ZttrZt25Jt
29uS///b+P8ADgEAAAMAAAABAEAAAAEBAAMAAAABACgAAAECAAMAAAADAAAEuAEDAAMAAAAB
gAUAAAEGAAMAAAABAAIAAAERAAQAAAABAAAACAESAAMAAAABAAEAAAEVAAMAAAABAAMAAAEW
AAMAAAABACgAAAEXAAQAAAABAAAEAgEaAAUAAAABAAAEvgEbAAUAAAABAAAExgEcAAMAAAAB
AAEAAAEoAAMAAAABAAIAAAAAAAAACAAIAAgSwAAAAAQAABLAAAAABAAA
ENDofIMG

if false
demo.show_image('tiff', 'pixarlog compressed', <<'ENDofIMG')
TU0AKgAAA3Z4nO2aIWzbUBCGz7HjYyVTNDAp0CgKDRorCauiSGFjRmWRxs0rlQ2FlUWKrLKQ
sIDK1BoyjFRUjZSF7f13eYnjpZOmdeuU3Wc1se+dn17uf/ee5SuRYRiGYRiGYRiGYRiGYRiG
YRg/YxVt3EG04IpLJkqocJ8VD6S1746ELsOP8UNMxO1J9MbDPQs2EvOKcV66uOsZoo34lwxb
JraMfEsif+S06Lrj74/5XNi2iG6jdbCKruMRz3gdfKUbF/OFi/c0vHbzfNvqBDl1giENKHdt
492dI/7sdIElMRV+i0l0G62igcx3ZMBhnUFc0ebV4fZzeBfDfx2ofyGr1DycOt+3/RXnALIB
63rTqnYo9SVYBxptVS11+WDz//WZh/MQ8Z5E8/AiwjdmPnTg9qfg1sV+Hn7gabiK3gWbSDPl
rcdsGIZhGIZhGIZhGIZhnCebXc1IKxeJ2LSGkbija+8G/zAXEd4LDqkTbFuo091T312l8iYd
79JhK7ngGY+dNSfoklKP38eX4VOMt73NN77Gr4B3tP4NOeLbCWY84qfYt18x4ntPUCd133XG
zv+RSOp5RD57DlRSiSo5Y59RY1HQV2fH1KRyunbDy/D/0VRjr3HVirQqMuJJxO2UvDY+R4hm
DKUe3RXywudIHbVrdBe8lNzJJfK08y2kQr5k7RfXS0ZeZaJNIfYl+35S8dF7j1fI8dHV8fmC
P7P23OO7+Dl8/idVvZCYz7hpz6VyNHKz/6ACxj8kqLNk5MWxIrkoktKhP0Re49uX+qtWxUtn
1cjP2Nvhi8/h7vyg7FDUXLKe+7o6tEKs/bgrGUEiFfdTeqji6qN1evSaiLYH34X4/piVaOnx
Qzx1Gr72f03MX9xdEdlcVh5dhZaMLBkS1XJCuZJxo1XXKOwWsOld6nPQbiC9asT70l+5+93Q
cMFY//rSk1eBaj7IgaoxW3wGlXu710X7QetwpxnRqTxSEHlEupJZk+x3v2xnIRk1VMvk3oJp
32tz7a2Du7svVNl0tUFdmtt+NiNWOuf1vE6+m/8aUR/5bSuTqBYu06/2UTjeLU7h8+h+r6nf
ZfAMMKYbUaTH3+Lj+1Tf+n4E/6K2c+kKqS26Py0k7xDTmcz+tPZ04VUmyTLvX4p/wdR4Gsm4
uV54beCT1Z5edCVNpc/TSn0HoOsh4gAOAQAAAwAAAAEAQAAAAQEAAwAAAAEAKAAAAQIAAwAA
AAMAAAQkAQMAAwAAAAGAjQAAAQYAAwAAAAEAAgAAAREABAAAAAEAAAAIARIAAwAAAAEAAQAA
ARUAAwAAAAEAAwAAARYAAwAAAAEAKAAAARcABAAAAAEAAANuARoABQAAAAEAAAQqARsABQAA
AAEAAAQyARwAAwAAAAEAAQAAASgAAwAAAAEAAgAAAAAAAAAIAAgACBLAAAAABAAAEsAAAAAE
AAA=
ENDofIMG
end

if false
demo.show_image('tiff', 'logluv compressed', <<'ENDofIMG')
TU0AKgAAAAgADgEAAAMAAAABAEAAAAEBAAMAAAABACgAAAECAAMAAAADAAAAtgEDAAMAAAAB
h3QAAAEGAAMAAAABAAIAAAERAAQAAAABAAAAAAESAAMAAAABAAEAAAEVAAMAAAABAAMAAAEW
AAMAAAABACgAAAEXAAQAAAABAAAAAAEaAAUAAAABAAAAvAEbAAUAAAABAAAAxAEcAAMAAAAB
AAEAAAEoAAMAAAABAAIAAAAAAAAAEAAQABASwAAAAAQAABLAAAAABAAA
ENDofIMG
end

demo.show_image('tiff', 'jpeg compressed', <<'ENDofIMG')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ENDofIMG

demo.show_image('tiff', 'deflate compressed', <<'ENDofIMG')
TU0AKgAAAah4nO2Y643EMAiEXQzNUIyboRiaoRh2ADuP+7N70knZkxhZlmNH2fkMRPGO0Wq1
Wq1Wq9Vqtb5L5m5qqi7Rm5kP4qdNfSSHc3dYDtfmMD8TQS3mn3b3TjRgm6YEgsaeYzwRg4TC
9PcjRMLYzpmxcABW8+ifNviJ6DIktLKN/vv3/xRHwXJuPoV5HTwqKK1Wq9VqtVqtz+VxwMjv
8ej/39fUpCGTdLLJVJmm6NlNQIQlev+AhyVMaDBfA6GYAU41nFrzqLSFSw3GjJf6ZeEpUvwu
bxDOcFREAkpXRKph7IjRzOjAvEnOC9jzcC7rnlp13YB29ip4wvxT1Jn+qwlthMQ5QWDbwlsE
RTis4uQX+cZl+zCPQfWWGZgsEkfHhaML3KyQN901lIjwL/6a4GvL5EEK4QErneKSKqPW0qaD
+awarnpJlkTIOnI9Aic1o1LsuEHuMdIyHTgrPyVTVKsM/YfuRzY4odxz3v2KBfaf021aDQqi
E2q3SYs0QxN1cdSOHnW0i8uWyYoR79dFIK95nedr5HhObRRmctXuUC9uNR31AAAOAQAAAwAA
AAEAQAAAAQEAAwAAAAEAKAAAAQIAAwAAAAMAAAJWAQMAAwAAAAGAsgAAAQYAAwAAAAEAAgAA
AREABAAAAAEAAAAIARIAAwAAAAEAAQAAARUAAwAAAAEAAwAAARYAAwAAAAEAKAAAARcABAAA
AAEAAAGfARoABQAAAAEAAAJcARsABQAAAAEAAAJkARwAAwAAAAEAAQAAASgAAwAAAAEAAgAA
AAAAAAAIAAgACBLAAAAABAAAEsAAAAAEAAA=
ENDofIMG

if Tk::PLATFORM['platform'] != 'windows'
demo.show_image('other', 'postscript', <<'ENDofIMG')
%!PS-Adobe-3
%%Title: postscript.ps
%%BoundingBox: 66 648 146 720
%%Pages: 1
%%DocumentProcSets: Adobe_distill 0.96
%%EndComments
%%BeginProcSet: Adobe_distill 0.96
/PROLOGUE 30 40 add dict def
 % 30 procedure entries + room for 40 cached font dictionaries
 PROLOGUE begin
 /clip { } def    % causes problems. remove if "clip" is needed
 /bdef { bind def } bind def	/ldef { load def } bdef
 /T { moveto show } bdef	/A { moveto ashow } bdef
 /W { moveto widthshow } bdef	/AW { moveto awidthshow } bdef
 /f /fill ldef			/R { { rlineto } repeat } bdef
 /r /rlineto ldef		/L { { lineto } repeat } bdef
 /m /moveto ldef		/l { moveto lineto stroke } bdef
 /x { 0 rlineto } bdef		/y { 0 exch rlineto } bdef
 /c /curveto ldef		/cp /closepath ldef
 /s /stroke ldef		/w /setlinewidth ldef
 /g /setgray ldef		/j /setlinejoin ldef
 /d /setdash ldef		/F /setfont ldef
 /C /setcmykcolor where { /setcmykcolor get }{ %ifelse
   { %def
     1 sub 3 { 3 index add neg dup 0 lt { pop 0 } if 3 1 roll } repeat
     setrgbcolor
   } bind
 } ifelse def
 /selectfont where { pop }{ %ifelse
     /selectfont { exch findfont exch scalefont setfont } bdef
 } ifelse
 /MF { exch findfont exch makefont setfont } bdef
 /FF /selectfont ldef
 /DF { selectfont currentfont def } bdef
 /BEGINPAGE { pop /pagesave save def } bdef
 /ENDPAGE { pop pagesave restore showpage } def
 /REMAP { %def
   FontDirectory 2 index known { pop pop pop } { %ifelse
     findfont dup length dict begin
       { 1 index /FID ne {def}{pop pop} ifelse } forall
       exch dup length 0 gt { /Encoding exch def }{ pop } ifelse
     currentdict end definefont pop
   } ifelse
 } bdef
 /RECODE { %def
    3 -1 roll 1 index findfont /Encoding get 256 array copy exch
    0 exch { %forall
     dup type/nametype eq
       { 3 {2 index} repeat put pop 1 add }{ exch pop }ifelse
    } forall pop 3 1 roll REMAP
 } bdef
 end %PROLOGUE
%%EndProcSet: Adobe_distill 0.96
%%EndProlog
%%BeginSetup
PROLOGUE begin
%%EndSetup
%%Page: 1 1
1 BEGINPAGE
1 g
2 setlinecap
144 53 m
390 x
-19 y
-390 x
eofill
0 g
1.268 w
0 setlinecap
4 setmiterlimit
[] 0 d
109.698 714.182 m
111.525 713.672 112.472 713.234 113.624 712.431 c
-22.3128 -25.8381 r
97.4019 682.652 103.794 672.84 100.329 664.844 c
102.034 675.687 90.7436 683.82 84.1099 684.842 c
25.588 29.3407 r
f
1.0361 w
f
f
91.8805 654.468 m
96.0255 654.285 100.155 654.188 104.481 654.188 c
113.804 654.188 122.752 654.636 131.141 655.44 c
134.222 655.737 136.724 658.133 137.108 661.279 c
137.79 666.855 138.107 672.696 138.107 678.705 c
138.107 684.714 137.79 690.555 137.108 696.131 c
136.724 699.277 134.222 701.674 131.141 701.97 c
127.476 702.321 123.704 702.604 119.84 702.813 c
119.84 702.813 103.842 668.373 102.231 664.827 c
99.8549 659.595 96.601 655.765 91.8805 654.468 c
f
0.7851 w
77.8202 655.44 m
74.7171 655.948 72.2887 657.716 71.8525 661.279 c
71.1725 666.855 70.8549 672.696 70.8549 678.705 c
70.8549 684.714 71.1725 690.555 71.8525 696.131 c
72.2381 699.277 74.7417 701.674 77.8202 701.97 c
82.1335 702.383 86.5932 702.702 91.1764 702.916 c
8.16241 9.07727 r
99.2863 711.98 101.266 712.345 102.996 711.921 c
-28.4564 -32.9334 r
80.6983 681.906 98.7446 677.354 98.7446 666.708 c
98.7446 655.332 86.3243 654.054 77.8202 655.44 c
f
0.0843 w
139.67 644.154 m
140.194 644.154 140.619 644.576 140.619 645.098 c
140.619 645.619 140.194 646.042 139.67 646.042 c
139.146 646.042 138.721 645.619 138.721 645.098 c
138.721 644.576 139.146 644.154 139.67 644.154 c
1 ENDPAGE
%%Trailer
end %PROLOGUE
%%Pages: 1
%%EOF
ENDofIMG
end

if false
demo.show_image('other', 'pdf', <<'ENDofIMG')
JVBERi0xLjIKJcfsj6IKNCAwIG9iago8PC9MZW5ndGggNSAwIFIvRmlsdGVyIC9GbGF0ZURl
Y29kZT4+CnN0cmVhbQp4nF1TSY7cQAy7+xX1AkFbaXlG3jBAgAAzh/z/Esp2uyeDPhhUayEp
FZMsnt/9/fg6/h6y/hzCHdS1UlyodH0dIrJJNyIWlLpE1OcLrGQOjLg6MOIm6+NoMULWiopN
bevz6HRiaUQ0aKMHW1L7itQitGA2UvwdDuhoIaw8zSN3UOVqTnILNLAiXQVy3NNvCvQs+M57
Zv4Gj6LivWK7UxS0dDCknBGl2jNY6MKoKmBkljwYfQUD2d8ZqjkKgINASEyEfCq208nc0Fon
YSdZIiFy7IldQmbACe4FraKUfVYkzEAgarhgIBJyvIGi+IYrCWqm4I6UJ4mvV4PmfTW4J3TI
tZCHQzdmDqeLY7Jgpf6IAG7KKVAsLAMBxWIEqrEvuAAc8xXp2Rtgkdhp038B7PfcCyQZ5WDV
oXIuWGdCI387SO/GjezVgcZyuRaAP1b3gXVmopb1ZfXXkSM/z6Imx+4xpHAsseGHgD2abFzu
2+qEzjwjl9fJN4vb6zd+ef1Ebq+fDrfZz4zH7GFhc0O32cPSJR+zXzIes0vnMPZlNk73ejWT
gFNDYs/IOM3tkTXmGUbMsaPF9cAwq8KuyNwxFMc4gJY+d47S0Td/68nSNxkiUNpz159H8Twg
g1LkcKyGBe6TkQYL3jhwTDwreSJ7gw9eZeBY3GZpPBU/FjY7/HX8A9+h3WJlbmRzdHJlYW0K
ZW5kb2JqCjUgMCBvYmoKNTE2CmVuZG9iagozIDAgb2JqCjw8Ci9UeXBlIC9QYWdlCi9NZWRp
YUJveCBbMCAwIDYxMiA3OTJdCi9QYXJlbnQgMiAwIFIKL1Jlc291cmNlcyA8PCAvUHJvY1Nl
dCBbL1BERl0KPj4KL0NvbnRlbnRzIDQgMCBSCj4+CmVuZG9iagoyIDAgb2JqCjw8IC9UeXBl
IC9QYWdlcyAvS2lkcyBbCjMgMCBSCl0gL0NvdW50IDEKPj4KZW5kb2JqCjEgMCBvYmoKPDwg
L1R5cGUgL0NhdGFsb2cgL1BhZ2VzIDIgMCBSCj4+CmVuZG9iago2IDAgb2JqCjw8IC9DcmVh
dGlvbkRhdGUgKEQ6MTk5NzEyMDUyMjU2MzcpCi9Qcm9kdWNlciAoQWxhZGRpbiBHaG9zdHNj
cmlwdCA1LjEwKQo+PgplbmRvYmoKeHJlZgowIDcKMDAwMDAwMDAwMCA2NTUzNSBmIAowMDAw
MDAwNzk5IDAwMDAwIG4gCjAwMDAwMDA3NDAgMDAwMDAgbiAKMDAwMDAwMDYyMCAwMDAwMCBu
IAowMDAwMDAwMDE1IDAwMDAwIG4gCjAwMDAwMDA2MDEgMDAwMDAgbiAKMDAwMDAwMDg0OCAw
MDAwMCBuIAp0cmFpbGVyCjw8IC9TaXplIDcgL1Jvb3QgMSAwIFIgL0luZm8gNiAwIFIKPj4K
c3RhcnR4cmVmCjkzOQolJUVPRgo=
ENDofIMG
end

#######################

Tk.mainloop
